#!/usr/bin/perl

########################################################################
# Copyright (c) 1996-2014 NetApp, Inc.
# All rights reserved
########################################################################

## command.pl : Launcher script to execute array specific scripts.

# input:
#
#        Relevant Element descriptions as follows:
#        Command::Name - operation to perform (discoverArrays, etc)
#        Command::OutputFile - file to write XML output data
#        Command::LogLevel - optional log level control, default is info.
#
#        Example:
#        <?xml version="1.0" encoding="ISO-8859-1"?>
#        <Command xmlns="http://www.vmware.com/srm/sra/v2">
#           <Name>discoverArrays</Name>
#           <OutputFile>discoverArrays.xml</OutputFile>
#           ...
#           <LogLevel>info</LogLevel>
#        </Command>
#
# output:
#
#        Debug/log/CLI output/CLI error written to STDOUT.
#
# return:
#     Exit Code: UNKNOWNERROR if OutputFile is NULL or missing in Input XML,
#                or failure to write to XML output.
#                SUCCESS, otherwise.
#                Note: other error such as Command::Name not found is reported
#                      in XML output ReturnCode.
# notes:
#     This script is the entry point for all commands accepted by the
#     array manager. Some commands are are:
#        discoverArrays: discovery of arrays.
#        discoverDevices: discovery of replicated LUNs.
#        failover: failover of LUNs.
#        testFailover (Start): creates a volume containing replicated data on
#           a secondary snapshot and ensures host visibility to it.
#        testFailover (Stop): terminates/cleanup the failover test.
#

# libraries loaded from SRM perl packages
use FindBin qw($Bin);    # Locate directory of original perl script
use XML::Simple qw(:strict);
use MIME::Base64;
use IO::Socket;
use File::Copy;
use Data::Dumper;
use File::Slurp;

# custom include folders for SRA perl libraries
use lib "$Bin/lib/SOAP/Transport";
use lib "$Bin/lib/CryptX";
use lib "$Bin/lib/SOAP";
use lib "$Bin/lib";
use lib "$Bin";

# libraries loaded from SRA perl packages
use SOAP::Lite;
use XML::Twig;
use Tie::IxHash;
use LWP::Protocol::https;

# the cryptography related libraries are loaded from vendor if they are not present in SRA lib folder
use Crypt::Mode::CBC;
use Crypt::PRNG;

use Tiny::OpenSSL::Key;
use Tiny::OpenSSL::Subject;
use Tiny::OpenSSL::Certificate;
use Tiny::OpenSSL::CertificateSigningRequest;
use Path::Tiny;
use Net::SSL::ExpireDate;

#File path for sra IP
our $IP_FILENAME = "$Bin/conf/ip.txt";

# File path for parms
our $PARMS_FILENAME = "$Bin/conf/keyring.properties";

# File path for keyring
# This file needs to be protected by user-ID, file permissions, etc.
our $KEYRING_FILENAME = "$Bin/conf/keyring.bin";

# File path for credential file
# This file should be protected by user/permissions but is not as important as keyring# Make sure this path is at least 16 characters since we use this for our IV.
our $CREDENTIAL_FILENAME = "$Bin/conf/./credentials.bin";

# Directory path to store sra self signed certificate
our $SRA_CERTIFICATE_DIRECTORY = "$Bin/conf/certificates";

# File path for sra self signed certificate
# This file is used for certificate based authentication when making calls from SRA adapter to OTV
our $SRA_CERTIFICATE = "$Bin/conf/certificates/sracert.crt";

# Key size in base-64 characters
# This will make AES-128 key
our $KEY_SIZE = 64;

# Password to protect keyring data. Unique per install.
# For this configuration, it should be 64 hex characters (32 bytes).
my $key_ring_pw = "";

# This is the IV (Initialization Vector) that is XOR'd on the first block (salt).
# For this configuration, it should by 32 hex characters (16 bytes) for AES
our $iv = "";

our $vp_username = "";
our $vp_password = "";

#Option for checking the version of the SRA
if ( ( defined( $ARGV[0] ) ) ) {
	if ( ( $ARGV[0] eq "-V" ) || ( $ARGV[0] eq "-v" ) ) {
		print("NetApp Storage Replication Adapter 9.13P2 for ONTAP");
	}
	elsif(  ( $ARGV[0] eq "-i" ) || ( $ARGV[0] eq "-I" ) || ( $ARGV[0] eq "-u" ) || ( $ARGV[0] eq "-U" )) {

		#Need the ip,username and password
		die "Usage: perl command.pl $ARGV[0] <sra-server-ip> <vp-username> <vp-password>\n" if @ARGV < 4;
		
		my $op = uc $ARGV[0];
		my @filenames =  ( $IP_FILENAME, $PARMS_FILENAME, $KEYRING_FILENAME, $CREDENTIAL_FILENAME );
		if( $op eq '-U'){
			#delete all config files 
			unlink @filenames;
		}else{
			#If its install, no file should exist
			die "SRA Adapter already configured. Use ".
			"\"perl command.pl -U <sra-server-ip> <vp-username> <vp-password>\" to update.\n" if (configDataExists());
		}
		my $SERVICE_UVAIP= "https://$ARGV[1]:9083";
		my $result = `wget --server-response --spider --quiet  $SERVICE_UVAIP/sramajorversion --no-check-certificate 2>&1 | awk 'NR==1{print $2}'`;
		#print "Response Code______:$result";
		my $substr = "200";
		if (index($result, $substr) != -1) {
 			print "SRA service status : Enabled";
		}
		else{
			print "Error: SRA service is either not available or not enabled";
			exit;
		}
		
		#Store the sra server ip in a separate file.
		storeSraServerIp($ARGV[1]);
		
		my $rc = 0;

		# Password to protect keyring data. Unique per install.
		# For this configuration, it should be 64 hex characters (32 bytes).
		$key_ring_pw = "";

		# This is the IV (Initialization Vector) that is XOR'd on the first block (salt).
		#For this configuration, it should by 32 hex characters (16 bytes) for AES
		$iv = "";

		unless ( ( -e $PARMS_FILENAME ) && ( !-z $PARMS_FILENAME ) ) {
			initParmsFile();
		}

		#print "\nLoading parameters file.\n";
		# Load our parameters
		open( PARMSFILE, "<$PARMS_FILENAME" );
		my @prop_fields;
		foreach $prop_line (<PARMSFILE>) {
			chomp($prop_line);

			#print "\nprop_line=$prop_line\n";
			@prop_fields = split( "=", $prop_line );

			#print "\nprop[0]=$prop_fields[0], prop[1]=$prop_fields[1]\n";
			if ( $prop_fields[0] eq 'key_ring_pw' ) {
				$key_ring_pw = $prop_fields[1];
			}
			elsif ( $prop_fields[0] eq 'salt' ) {
				$iv = $prop_fields[1];
			}
		}
		close PARMSFILE;

		#print("Creating Symmetric Key");
		unless ( ( !-z $KEYRING_FILENAME ) && ( -e $KEYRING_FILENAME ) ) {
			initSymmetricKey( $key_ring_pw, $iv );
		}

		# After creating a key then encrypt the credentials

		# Load our parameters
		open( PARMSFILE, "<$PARMS_FILENAME" );
		my @prop_fields;
		foreach $prop_line (<PARMSFILE>) {
			chomp($prop_line);

			#print "\nprop_line=$prop_line\n";
			@prop_fields = split( "=", $prop_line );

			#print "\nprop[0]=$prop_fields[0], prop[1]=$prop_fields[1]\n";
			if ( $prop_fields[0] eq 'key_ring_pw' ) {
				$key_ring_pw = $prop_fields[1];
			}
			elsif ( $prop_fields[0] eq 'salt' ) {
				$iv = $prop_fields[1];
			}
		}
		close PARMSFILE;

		open( KEYRING, "<$KEYRING_FILENAME" );
		my $bh4 = "";
		foreach $kr_line (<KEYRING>) {
			$b64 .= $kr_line;
		}
		close KEYRING;

		#print "\nBase64 in: $b64\n";
		my $enc_input = decode_base64($b64);

		my $cbc = Crypt::Mode::CBC->new('AES');
		$symmetric_key = $cbc->decrypt(
			$enc_input,
			pack( "H*", $key_ring_pw ),
			pack( "H*", $iv )
		);
		chomp($symmetric_key);

		unless ( ( !-z $CREDENTIAL_FILENAME ) && ( -e $CREDENTIAL_FILENAME ) ) {
			storeCredentials( $symmetric_key, $iv, $ARGV[2], $ARGV[3] );
		}
		
		#Generate the ssl certificate which will be used for authentication
		generateAndRegisterSelfSignedCertificate();
		
		#Set appropriate permissions
		chown 'srm', 'srm', @filenames;
		chmod 0755, @filenames;
		
	
	}
	elsif ( $ARGV[0] =~ m/^decrypt/ ) {
		loadKeyRing();
		decryptStoredCreds( $symmetric_key, $iv );
		print( " Username: " . $vp_username . ", password ******" );
	}
	else {
		print("Not a valid argument use -v or -V ");
		print("usage: perl command.pl -v");
	}
	exit;
}

my $currentLogLevel = Info;
my @output;
my $returnCode;
use constant XMLNS => "http://www.vmware.com/srm/sra/v2";

# XML output Element and Attribute labels
my $XML_DECLARATION       = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
my $XML_RESPONSE_ELEMENT  = "Response";
my $XML_ERRORCODE_ELEMENT = "Error";
my $outputFile;
my $loglevel;
my $currentLogLevel = Info;

# It's important to set ForceArray=>1 to force nested elements as arrays,
# before passing to CLI wrapper script. Otherwise, CLI wrapper will not
# be able to easily read single nested element.
my $xml = new XML::Simple(
	KeyAttr    => [],
	ForceArray => 1,
	KeepRoot   => 1
);
my $incoming_XML = eval { $xml->XMLin('-') };   # read XML input data from STDIN

my $workingxml = $xml->XMLout($incoming_XML)
  ; # Extracting xml and putting it in a vraiable so that it can begave as string .Incoming xml from SRM  is stored in workingxml for future use

my $vp_url = getSraServerIp();
print("vp_url =$vp_url ");

Log_output( Verbose, $currentLogLevel, "sra_server_url =$vp_url" );

my $logLevelString = $incoming_XML->{Command}[0]->{LogLevel}[0];
if ( $logLevelString eq "" ) {
	$currentLogLevel = Quiet;
	Log_output( Quiet, $currentLogLevel,
		"Missing LogLevel parameter in XML input" );
	commandUsage();
	my_exit(UNKNOWNERROR);
}
$currentLogLevel = findLogLevel($logLevelString);
$outputFile      = $incoming_XML->{Command}[0]->{OutputFile}[0];
Log_output( Quiet, $currentLogLevel, "outputFile =$outputFile" );
print("outputFile =$outputFile");

if ( $outputFile eq "" ) {
	Log_output( Quiet, $currentLogLevel,
		"Missing OutputFile parameter in XML input." );
	commandUsage();
	my_exit(UNKNOWNERROR);
}

# Truncate the existing output file (to avoid including previous result
# for failoverStop test)
truncate( $outputFile, 0 );

my $command =
  $incoming_XML->{Command}[0]->{Name}[0];    # extracting my_exitmy_exit
Log_output( Quiet, $currentLogLevel, "command =$command" );

if ( !(configDataExists())  && ( $command eq 'queryInfo' ) ) {
	$returnCode = SUCCESS;
	queryInfo();
	my_exit($returnCode);
}

loadKeyRing();
decryptStoredCreds( $symmetric_key, $iv );
Log_output( Verbose, $currentLogLevel, "vp_username =$vp_username" );
#Log_output( Verbose, $currentLogLevel, "vp_password =$vp_password" );

#code for making SOAP connection
$ENV{PERL_LWP_SSL_VERIFY_HOSTNAME} = 0;
my $SERVICE_LOC = 'https://10.60.170.211:9083';
my $SERVICE_LOC = $vp_url;
my $SERVICE_NS  = $SERVICE_LOC . "\/"
  . "services/vasa-public_0_1?wsdl=VasaPublicAPIPortType.wsdl";
my $SRA = "sra";

#adding VP credential
my $params = SOAP::Data->name(
	"credential" => \SOAP::Data->value(
		SOAP::Data->name( "VPPassword"      => $vp_password ),
		SOAP::Data->name( "VPUsername"      => $vp_username ),
		SOAP::Data->name( "applicationName" => $SRA )
	)
);

my $query = SOAP::Data->name("command")->value($workingxml)->type('string');
my $service =
  SOAP::Lite->proxy( $SERVICE_NS, ssl_opts => [ SSL_verify_mode => 0 ] );

# printing the xml output and storing it to service.txt
open( FILE_SOAP, ">> SOAPConnection.txt" );
print FILE_SOAP Dumper($service);
close(FILE_SOAP);

# if SRM sends any wrong/ambigous  command , it should be checked and filtered

#if ($command =~ /^(queryCapabilities|queryInfo|queryConnectionParameters|queryErrorDefinitions|queryStrings|discoverArrays|discoverDevices|testFailoverStart|failover| estFailoverStop|syncOnce|querySyncStatus|reverseReplication|prepareReverseReplication|prepareRestoreReplication|restoreReplication|queryReplicationSettings|prepareFailover|failover)$/ )
if ( grep { $command eq $_ }
	qw(queryInfo queryCapabilities queryConnectionParameters queryErrorDefinitions queryStrings discoverArrays discoverDevices testFailoverStart failover testFailoverStop syncOnce querySyncStatus reverseReplication prepareReverseReplication prepareRestoreReplication restoreReplication queryReplicationSettings prepareFailover failover)
  )
{
	#Check if the certificate is valid. 
	if ( ( !-z $SRA_CERTIFICATE ) && ( -e $SRA_CERTIFICATE ) ) {
		$ed = Net::SSL::ExpireDate->new( file  => './conf/certificates/sracert.crt' );
		if($ed->is_expired) {
			print "\nALERT: The self signed certificate has been expired. Regenerating it and registering it at the OTV backend\n";
			generateAndRegisterSelfSignedCertificate();
		}

		if($ed->is_expired('7 days')) {
			print "\nALERT: The certificate will expire in next 7 days. Regenerating it and registering it at the OTV backend\n";
			generateAndRegisterSelfSignedCertificate();
		} 
	} else {
		# This is the first time you are running the script after installation. Hence the self signed certificate is not generated.
		print "\nThe self signed certificate file doesn't exist. Generating it and registering the same at the OTV backend.\n";
		generateAndRegisterSelfSignedCertificate();
	}
	my $cert_as_string = read_file('./conf/certificates/sracert.crt');
    my $certificate = SOAP::Data->name("certificate")->value($cert_as_string)->type('string');
	
	#calling sraLaunchTask at VP through SOAP
	my $retry = 0;
	my $ID;    #= $t->{id};# this will store the ID returned by sraLaunchTask
	my $status;
	while ( $retry < 5 ) {
		my $som = $service->call( 'sraLaunchTask', $params, $query, $certificate );

		## XML Parsing from sraLaunchTask
		for my $t ( $som->valueof('//sraLaunchTaskResponse/return') ) {

			$status = $t->{Status};

		}

		if ( $status eq 'IN_PROGRESS' ) {

			#$retry=5; #no more retry as VP has processed the SRALAUNCH TASK
			for my $t ( $som->valueof('//sraLaunchTaskResponse/return') ) {

				$ID = $t->{id};    #storing the ID
				print("\n task id is $ID");
				$status = $t->{Status};
				print $t->{id} . " - "
				  . $t->{outgoingObject} . "  - "
				  . $t->{Status} . " - " . "\n";

			 # printing the xml output and storing it to sraLaunchTaskReturn.txt
				print("\n sraLaunchTask executed without errors \n");
				open( FILE_SRA_LAUNCH, ">> sraLaunchTaskReturn.txt" );
				print FILE_SRA_LAUNCH (
					"\n sraLaunchTask executed without errors \n");
				close(FILE_SRA_LAUNCH);
			}
			last;

		}
		else {
			print(" \n sraLaunchTask executed with errors \n");
			$retry++;
			if ( $retry <= 4 ) {
				print(
"\n SRA Server is unable to process SRALAUNCH TASK. Retrying ...Retry count :$retry"
				);

# this sleep is not needed. Sleep is only needed in a poll
# otherwise there are errors processing commands, VP to SRA needs a small break between SOAP API calls
#sleep(10);

			}
			else {
				print(
"\n SRA Server is unable to process SRALAUNCH TASK. No more retry .Permanent Error"
				);
				exit();
			}
			open( FILE_SRA_LAUNCH, ">> sraLaunchTaskReturn.txt" );
			print FILE_SRA_LAUNCH (" \n sraLaunchTask executed with errors \n");
			print FILE_SRA_LAUNCH Dumper($som);
			close(FILE_SRA_LAUNCH);
		}
	}
	Log_output( Quiet, $currentLogLevel, "taskID =$ID" );

	print("\n task id to be sent for sraPollOnTask is $ID\n");
	my $params4 = SOAP::Data->name("taskid")->value($ID)->type('string')
	  ;    #creating the parameter with the ID to be send by sraPollOnTask
	my $retry = 0;
	while ( $retry < 4000 ) {
		sleep(10);
		my $som3 = $service->call( 'sraPollOnTask', $params, $params4, $certificate )
		  ;    # calling sraPollOnTask

#print Dumper($som3->result);
# printing the xml output of sraPollOnTask and storing it to sraPollOnTaskReturn.txt
		if ( open( FILE_SRA_POLL, ">> sraPollOnTaskReturn.txt" ) ) {
			print FILE_SRA_POLL Dumper($som3);
		}
		else {

		}
		close(FILE_SRA_POLL) or do { };

		##XML parsing From sraPollOnTask
		my $logMessages
		  ;    # variable to store log message returned by sraPollOnTask
		my $err_srapollon;
		for my $t ( $som3->valueof('//sraPollOnTaskResponse/return') ) {
			print "@{$t->{LogMessages}}";
			$logMessages = $t->{LogMessages};

			#$ID=$t->{id};
			$status = $t->{Status};

#print $t->{id} . " - " . $t->{ProgressUpdate} . "  - " . $t->{Status} . " - " . "\n" ;
			print("\n STATUS = $t->{Status} \n");
			print("\n ID = $t->{id} \n");
		}
		$retry++;

	  #VP response xml which needs to be restructured before sending back to SRM
		my $som44 = $som3->valueof('//return/NetApp_VP_Response');

#Restructuring the xml response from VP
#this will work for other commands also if VP generates same type of xml 	, otherwise here change might be needed.
		$som44 =~ s/sra://g;
		$som44 =~ s/<Response>//g;
		$som44 =~ s/<\/NetApp_VP_Response>//g;
		$som44 =~ s/<NetApp_VP_Response/<Response/g;

		#print($som44);
		$som44 =~ s/\<\/Response\>\<\/Response\>/\<\/Response\>/g;
		$som44 =~ s/xmlns\:sra="http\:\/\/www\.vmware\.com\/srm\/sra\/v2"//g;
		$som44 =~ s/www.netapp.com/www.vmware.com\/srm/g;

		#print Dumper($som44);

		my $t = XML::Twig->new(
			keep_atts_order => 1,
			keep_encoding   => 1,
			keep_spaces     => 1,
			twig_handlers =>
			  { _all_ => sub { $_->set_empty() if !$_->has_child(); }, },
			empty_tags => 'html'
		);

		if ( defined($som44) ) {
			$t->parse($som44);
		}
		my $root = $t->root;
		if ( defined($root) ) {
			$root->set_tag('Response');

			# writing XML to the unique output file expected by VMware
			open( $OUTPUTXML, '>>', $outputFile );
			print $OUTPUTXML $XML_DECLARATION;
			$t->root->print( \*$OUTPUTXML );

	   # writing XML for the latest command to the file path below for debugging
			open( SRAOUTPUT, '>>' . $Bin . '/output.xml' );
			print SRAOUTPUT \$outputfile . "\n";
			print SRAOUTPUT $XML_DECLARATION;
			$t->root->print( \*SRAOUTPUT );
			close(SRAOUTPUT);

#to show each tag in new line we are calling xmlin and xmlout once again ,we can ignore it and write the xml directlyy in $outputFile

#my $sra_response_xml = $xml->XMLin('C:\Program Files\VMware\VMware vCenter Site Recovery Manager\storage\sra\CMODE_ONTAP\output.xml');
#my $xmlOutput = eval {
#$xml->XMLout($sra_response_xml,
#     KeyAttr    => [],
#     xmldecl    => $XML_DECLARATION,
#     OutputFile => $outputFile,
#     rootname   => 'Response');
#};
			last;
		}
		else {
			print(" sraPollOnTask COMPLETED with an error \n");
			if ( open( FILE_SRA_POLL, ">> sraPollOnTaskReturn.txt" ) ) {
				print FILE_SRA_POLL (
					"\n sraPollOnTask COMPLETED with errors \n");
				print FILE_SRA_POLL Dumper($som3);
			}
			else {

			}
			close(FILE_SRA_POLL) or do { };
		}

		if ( $status eq 'COMPLETE' ) {
			print("sraPollOnTask COMPLETED without an error \n");

			if ( open( FILE_SRA_POLL, ">> sraPollOnTaskReturn.txt" ) ) {
				print FILE_SRA_POLL (
					"\n sraPollOnTask COMPLETED without an error \n");
				print FILE_SRA_POLL Dumper($som3);
			}
			else {

			}
			close(FILE_SRA_POLL) or do { };
		}
		else {
			print(" sraPollOnTask COMPLETED with an error \n");

			if ( open( FILE_SRA_POLL, ">> sraPollOnTaskReturn.txt" ) ) {
				print FILE_SRA_POLL (
					"\n sraPollOnTask COMPLETED with errors \n");
				print FILE_SRA_POLL Dumper($som3);
			}
			else {

			}
			close(FILE_SRA_POLL) or do { };
		}

		myLogRotate("sraPollOnTaskReturn.txt");
	}

}
else {
	if ( $command eq '' ) {
		Log_output( Quiet, $currentLogLevel,
			"Command is not specified in xml input." );
	}
	else {
		Log_output( Quiet, $currentLogLevel,
			"Unrecognized command: $command." );
	}
	commandUsage();
	my $response;
	$response = {
		$XML_RESPONSE_ELEMENT => [
			{
				xmlns => XMLNS,
				$XML_ERRORCODE_ELEMENT => [ { code => COMMANDNOTFOUND }, ]
			}
		]
	};

	my $xmlOutput = eval {
		$xml->XMLout(
			$response,
			KeyAttr    => [],
			xmldecl    => $XML_DECLARATION,
			KeepRoot   => 1,
			OutputFile => $outputFile
		);
	};
	if ($@) {
		Log_output( Quiet, $currentLogLevel,
			"Failed to create final XML output:\n$@" );
		my_exit(UNKNOWNERROR);
	}

	my_exit(SUCCESS);
} ## end else [ if ($command eq 'queryInfo')

my $perlPath = "\"" . $^X . "\"";

if ( !defined($perlPath) ) {

	#open failed -
	Log_output( Quiet, $currentLogLevel, "Current Perl path not found" );
}

#my_exit(SUCCESS);    #Since the command.pl is called successfully

#

# Command.pl command options
#
## @todo the DTD URL will need to be changed.
sub commandUsage {
	Log_output( Quiet, $currentLogLevel, "Input xml doesnot confirm to DTD" );
}

sub Log_output() {
	my $msgLogLevel     = shift;
	my $currentLogLevel = shift;
	my $msg             = shift;
	$msg = $msg . "\n";
	if ( $currentLogLevel >= $msgLogLevel ) {
		push( @output, $msg );
		print("Log_output : output = @output");

	}
}

# Function to generate ssl self signed certificate and register the same at the OTV backend.
sub generateAndRegisterSelfSignedCertificate {
	#print("\nGenerating self signed certificate\n");

	if(!-d $SRA_CERTIFICATE_DIRECTORY) {
		mkdir $SRA_CERTIFICATE_DIRECTORY || die "Failed to create directory!";
	} 

	my $key = Tiny::OpenSSL::Key->new( file => path('./conf/certificates/srakey.key') );
	$key->create;

	### $key
	my $subject = Tiny::OpenSSL::Subject->new(
		commonname          => 'SRA certificate',
		organizational_unit => 'NetApp',
		organization        => 'HCE',
		locality            => 'San Jose',
		state               => 'California',
		country             => 'US'
	);

	### $subject
	my $cert = Tiny::OpenSSL::Certificate->new(
		file    => path('./conf/certificates/sracert.crt'),
		subject => $subject,
		key     => $key
	);

	### $cert
	my $csr = Tiny::OpenSSL::CertificateSigningRequest->new(
		file    => path('./conf/certificates/sracert.csr'),
		key     => $key,
		subject => $subject
	);

	### $csr
	$csr->create;
	$cert->self_sign($csr);
	
	#Waiting for few seconds before sending this certificate to SRA Server to avoid certificate timing issue
	sleep(5);
	
	#print("\nGenerated the self signed certificate successfully\n");
	
	print("\nConfiguring the SRA adapter .... \n");
	my $vp_url = getSraServerIp();
	my $SERVICE_LOC = $vp_url;
	my $SERVICE_NS  = $SERVICE_LOC . "\/"
						. "services/vasa-public_0_1?wsdl=VasaPublicAPIPortType.wsdl";
	my $SRA = "sra";
	decryptStoredCreds( $symmetric_key, $iv );

	#adding VP credential
	my $params = SOAP::Data->name(
		"credential" => \SOAP::Data->value(
			SOAP::Data->name( "VPPassword"      => $vp_password ),
			SOAP::Data->name( "VPUsername"      => $vp_username ),
			SOAP::Data->name( "applicationName" => $SRA )
		)
	);
	
	# Obtaining the certificate as a pem string
	my $cert_as_string = read_file('./conf/certificates/sracert.crt');
	my $certificate = SOAP::Data->name("certificate")->value($cert_as_string)->type('string');
	
	# Invoking sraLoginTask to OTV server
	#code for making SOAP connection
	$ENV{PERL_LWP_SSL_VERIFY_HOSTNAME} = 0;
	my $service =
	SOAP::Lite->proxy( $SERVICE_NS, ssl_opts => [ SSL_verify_mode => 0 ] );

	my $somLogin = $service->call( 'sraLoginTask', $params, $certificate );
	
	# Reading the returned from the above value
	my $loginResult = $somLogin->valueof('//sraLoginTaskResponse/return');
	
	if ( $loginResult eq 'success' ) {
		print("\nSucess: Successfully configured the SRA adapter. \n");
	} 
	else {
		print("\nError: Failed to configure the SRA adapter. Verify if the username and password are correct and try again. Please refer to sra logs at the backend if the issue persists. \n");
		#delete all config files 
		unlink @filenames;
		exit();
	}
}

##########################################
# Rotate log such that 10 versions       #
# of file are maintained of 100MB each   #
##########################################
sub myLogRotate($) {

	my $logfile = shift;
	my $maxsize = 100000000;
	my $maxlogs = 10;

	### Check Size of log
	if ( -s $logfile > $maxsize ) {
		### Rotate old logs
		for ( my $i = $maxlogs - 1 ; $i > 0 ; $i-- ) {
			if ( -e "${logfile}.$i" ) {
				my $j = $i + 1;
				rename "${logfile}.$i", "${logfile}.$j" or do { };
			}
		}
		rename "$logfile", "${logfile}.1" or do { };

		open( FILEHANDLE, '>' . $logfile ) or do { };
		close(FILEHANDLE) or do { };

		if ( -e "${logfile}.$maxlogs" ) {
			unlink("${logfile}.$maxlogs") or do { };
		}
	}
}

sub my_exit() {
	$returnCode = shift;
	print "<Response>";
	print @output;
	print "</Response>";
	exit($returnCode);
}

#############################
# Log this message              #
# based on log level passed #
#############################
sub nss_Log($$$) {
	my $msgLogLevel     = shift;
	my $currentLogLevel = shift;
	my $msg             = shift;

	#To take the local time via function localtime
	my ( $sec, $min, $hour, $day, $month, $year, @other_parameter ) =
	  localtime(time);

	#Prints the current local date and time ex: "31-6-2011-22:29:44"
	if ( $day < 10 )   { $day   = "0" . $day; }
	if ( $month < 10 ) { $month = "0" . $month; }
	my $date_time_stamp = "$day-"
	  . ++$month . "-"
	  . ( $year + 1900 ) . "T"
	  . sprintf( "%02d", $hour ) . ":"
	  . sprintf( "%02d", $min ) . ":"
	  . sprintf( "%02d", $sec ) . "";

	if ( $currentLogLevel >= $msgLogLevel ) {
		print STDOUT "\n$date_time_stamp  $msg ";
	}

} ## end sub nss_Log($$$)

#########################################
# Log SRA Signature                                                        #
# based on Version, Build Date, Package #
#########################################
sub nss_Log_Signature() {

	nss_Log( Quiet, $currentLogLevel, $SRA_build_info );
	if ($@) {
		$returnCode = UNKNOWNERROR;
		nss_Log( Quiet, $currentLogLevel, "Error Writing XML output" );
	}
}

sub findLogLevel($) {
	my $level = shift;
	if ( $level =~ /Quiet/i ) {
		return Quiet;
	}
	elsif ( $level =~ /Error/i ) {
		return Error;
	}
	elsif ( $level =~ /Warning/i ) {
		return Warning;
	}
	elsif ( $level =~ /Info/i ) {
		return Info;
	}
	elsif ( $level =~ /Verbose/i ) {
		return Verbose;
	}
	elsif ( $level =~ /Trivia/i ) {
		return Trivia;
	}
	else {
		#default is info
		return Info;
	}
} ## end sub findLogLevel($)

##########################################
# print SRA queryInfo#
##########################################

sub queryInfo() {
	my $queryInfoFilePath = "$FindBin::RealBin/queryInfo.xml";
	my $queryxml          = new XML::Simple(
		KeyAttr    => [],
		ForceArray => 1,
		KeepRoot   => 1
	);
	my $responseRef = eval {
		$responseRef =
		  $queryxml->XMLin($queryInfoFilePath);    # parse queryInfo xml file
	};
	if ($@) {
		$returnCode = UNKNOWNERROR;
		nss_Log( Quiet, $currentLogLevel,
			"Error reading queryInfo xml file : $@" );
		return;
	}

	#print "output file is $outputfile \n";
	my $xmlOutput = eval {
		$xml->XMLout(
			$responseRef,
			KeyAttr    => [],
			xmldecl    => $XML_DECLARATION,
			OutputFile => $outputFile,
			KeepRoot   => 1,
		);
	};

	if ($@) {
		$returnCode = UNKNOWNERROR;
		nss_Log( Quiet, $currentLogLevel, "Error Writing XML output" );
	}
} ## end sub queryInfo()

# This method creates a new symmetric key for any kind of encrypt/decrypt operations
sub initSymmetricKey {
	#
	# This file needs to be locked down as well as possible using OS permissions
	# and file system attributes.
	# Attributes to use: user-only read-write, System file, Hidden file, etc.
	# Optional attributes: System, Hidden
	#

	# Initializing symmetric key
	my $key_ring_pw = $_[0];
	my $iv          = $_[1];

	# Our symmetric crypto key.  Unique per install.
	# Lose this and lose your data!

	# Initialize our symmetric key for any kind of encrypt/decrypt operations
	my $symmetric_key =
	  Crypt::PRNG::random_string_from( '0123456789ABCDEFHOLLYPOWER', 64 );

	# Unique key is [$symmetric_key]

	# Create keyring payload
	my $cbc = Crypt::Mode::CBC->new('AES');

	# The payload can be more complex to hold more than 1 key, if needed.
	# Added complexity makes it harder to parse the file back into fields.
	my $payload = "$symmetric_key\n";
	my $enc_payload =
	  $cbc->encrypt( $payload, pack( "H*", $key_ring_pw ), pack( "H*", $iv ) );

	# Create a new keyring file.
	open( KEYRING, ">$KEYRING_FILENAME" );
	my $b64 = encode_base64($enc_payload);
	print KEYRING "$b64";

	close KEYRING;
}

sub initParmsFile {
	#
	# This file needs to be locked down as well as possible using OS permissions
	# and file system attributes.
	# Attributes to use: user-only read-write, System, Hidden
	#

	# Generate a new parms file.

	# Password to protect keyring data. Unique per install.
	# For this configuration, it should be 64 hex characters (32 bytes).
	my $key_ring_pw =
	  Crypt::PRNG::random_string_from( '0123456789ABCDEFHollyPOWER', 64 );

	#print "\nOur NEW keyring password: [$key_ring_pw]\n";

# This is the IV (Initialization Vector) that is XOR'd on the first block (salt).
# For this configuration, it should by 32 hex characters (16 bytes) for AES
	my $iv =
	  Crypt::PRNG::random_string_from( '0123456789ABCDEFHollyPOWER', 32 );

	#print "\nOur NEW IV is: [$iv]\n";

	open( PARMSFILE, ">$PARMS_FILENAME" );
	print PARMSFILE "key_ring_pw=$key_ring_pw\n";
	print PARMSFILE "salt=$iv\n";
	close PARMSFILE;
}

sub storeCredentials {

	my $symmetric_key = $_[0];
	my $iv            = $_[1];
	my $VP_USER       = $_[2];
	my $password      = $_[3];

	# Now write our password out to the keyring
	my $cbc     = Crypt::Mode::CBC->new('AES');
	my $payload = "$VP_USER:$password";

	my $enc_payload = $cbc->encrypt( $payload, pack( "H*", $symmetric_key ),
		pack( "H*", $iv ) );

	# Create a new credential file.
	open( CREDFILE, ">$CREDENTIAL_FILENAME" );
	my $b64 = encode_base64($enc_payload);
	print CREDFILE "$b64";
	#print "\nBase64 out: $b64\n";
	close CREDFILE;

	#print("Stored credentials successfully!\n");
}

sub loadKeyRing {

	# Load our parameters
	open( PARMSFILE, "<$PARMS_FILENAME" );
	my @prop_fields;
	foreach $prop_line (<PARMSFILE>) {
		chomp($prop_line);

		#print "\nprop_line=$prop_line\n";
		@prop_fields = split( "=", $prop_line );

		#print "\nprop[0]=$prop_fields[0], prop[1]=$prop_fields[1]\n";
		if ( $prop_fields[0] eq 'key_ring_pw' ) {
			$key_ring_pw = $prop_fields[1];
		}
		elsif ( $prop_fields[0] eq 'salt' ) {
			$iv = $prop_fields[1];
		}
	}
	close PARMSFILE;

	# Our symmetric crypto key.  Unique per install.
	# Lose this and lose your data!
	$symmetric_key = "";

	print "\nLoading symmetric key from keyring\n";
	open( KEYRING, "<$KEYRING_FILENAME" );
	my $bh4 = "";
	foreach $kr_line (<KEYRING>) {
		$b64 .= $kr_line;
	}
	close KEYRING;

	#print "\nBase64 in: $b64\n";
	my $enc_input = decode_base64($b64);

	my $cbc = Crypt::Mode::CBC->new('AES');
	$symmetric_key = $cbc->decrypt( $enc_input, pack( "H*", $key_ring_pw ),
		pack( "H*", $iv ) );
	chomp($symmetric_key);

}

sub decryptStoredCreds {

	# Only if the file exists
	if ( ( !-z $CREDENTIAL_FILENAME ) && ( -e $CREDENTIAL_FILENAME ) ) {
		my $symmetric_key = $_[0];
		my $iv            = $_[1];
		my $b64;

		# Credential retrieval
		#print "\nLoading credentials\n";
		open( CREDFILE, "<$CREDENTIAL_FILENAME" );
		$b64 = "";
		foreach $kr_line (<CREDFILE>) {
			$b64 .= $kr_line;
		}
		close CREDFILE;

		$enc_input = decode_base64($b64);

		$cbc   = Crypt::Mode::CBC->new('AES');
		$input = $cbc->decrypt(
			$enc_input,
			pack( "H*", $symmetric_key ),
			pack( "H*", $iv )
		);

		my $colon = index( $input, ':' );
		$vp_username = substr( $input, 0, $colon );
		$vp_password = substr( $input, $colon + 1 );

	}
	else {
		print "Credential file does not exist";
	}
}

sub storeSraServerIp(){
	my $sra_ip = "https://".$_[0].":9083";
	
	#print ("\nSaving SRA Server IP: ".$sra_ip."\n\n");
	open( IPFILE, ">$IP_FILENAME" );
	print IPFILE $sra_ip;
	close(IPFILE);
}

sub getSraServerIp(){
	if ( ( !-z $IP_FILENAME ) && ( -e $IP_FILENAME ) ) {
		open( IPFILE, "<$IP_FILENAME" );
		my $sra_ip = <IPFILE>;
		chomp($sra_ip);
		print ("\nSaving SRA Server IP".$sra_ip);
		close(IPFILE);
		return $sra_ip;
	}
	else{
		print("SRA Server IP file does not exist.");
		return -1;
	}
	
}

sub configDataExists(){
	return ((-e  $IP_FILENAME) || (-e  $PARMS_FILENAME) || (-e  $KEYRING_FILENAME) || (-e  $CREDENTIAL_FILENAME));
}
